import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:viserpay_merchant/core/utils/debouncer.dart';
import 'package:viserpay_merchant/data/services/api_service.dart';
import 'package:viserpay_merchant/view/components/divider/custom_divider.dart';
import 'package:viserpay_merchant/view/screens/invoice/create_invoice/widget/invoice_items.dart';

import '../../../../../core/utils/dimensions.dart';
import '../../../../../core/utils/my_color.dart';
import '../../../../../core/utils/my_strings.dart';
import '../../../../../core/utils/style.dart';
import '../../../../../data/controller/invoice/create_invoice_controller.dart';
import '../../../../../data/model/invoice/create_invoice_response_model.dart';
import '../../../../components/bottom-sheet/bottom_sheet_header_row.dart';
import '../../../../components/bottom-sheet/custom_bottom_sheet.dart';
import '../../../../components/buttons/rounded_button.dart';
import '../../../../components/card/bottom_sheet_card.dart';
import '../../../../components/text-form-field/custom_text_field.dart';
import '../../../../components/text/label_text.dart';
import '../../../transaction/widget/filter_row_widget.dart';

class InvoiceDetails extends StatefulWidget {
  const InvoiceDetails({Key? key}) : super(key: key);

  @override
  State<InvoiceDetails> createState() => _InvoiceDetailsState();
}

class _InvoiceDetailsState extends State<InvoiceDetails> {
  final formKey = GlobalKey<FormState>();
  final MyDebouncer _debouncer = MyDebouncer(delay: Duration(milliseconds: 500));

  @override
  Widget build(BuildContext context) {
    return GetBuilder<CreateInvoiceController>(
        builder: (controller) => Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                CustomTextField(
                  needOutlineBorder: true,
                  labelText: MyStrings.invoiceTo.tr,
                  hintText: MyStrings.enterUsernameOrMobileNumber.tr,
                  controller: controller.invoiceToController,
                  onChanged: (String query) {
                    _debouncer.run(() {
                      // This function is called after the delay
                      print("Searching for: $query");
                      // You can call your search API or function here

                      controller.checkUserExist(query);
                    });
                  },
                  validator: (value) {
                    if (value!.isEmpty) {
                      return MyStrings.invoiceToEmptyMsg.tr;
                    } else {
                      return null;
                    }
                  },
                ),
                const SizedBox(height: Dimensions.space10),
                if (controller.userExist) ...[
                  Text.rich(
                    TextSpan(
                      children: [
                        TextSpan(
                          text: "${MyStrings.name}: ",
                          style: regularDefault.copyWith(
                            color: MyColor.getTextColor(),
                            fontWeight: FontWeight.bold, // Makes title bold
                          ),
                        ),
                        TextSpan(
                          text: "${controller.existUserModal?.data?.user?.firstname} ${controller.existUserModal?.data?.user?.lastname}",
                          style: regularDefault.copyWith(color: MyColor.getTextColor()),
                        ),
                      ],
                    ),
                  ),
                  Text.rich(
                    TextSpan(
                      children: [
                        TextSpan(
                          text: "${MyStrings.phone}: ",
                          style: regularDefault.copyWith(
                            color: MyColor.getTextColor(),
                            fontWeight: FontWeight.bold, // Makes title bold
                          ),
                        ),
                        TextSpan(
                          text: "+${controller.existUserModal?.data?.user?.dialCode}${controller.existUserModal?.data?.user?.mobile}",
                          style: regularDefault.copyWith(color: MyColor.getTextColor()),
                        ),
                      ],
                    ),
                  ),
                ] else if (controller.invoiceToController.text.trim().isNotEmpty) ...[
                  Text(
                    "${MyStrings.userNotFound}",
                    style: regularDefault.copyWith(color: MyColor.colorRed),
                  ),
                ],
                CustomDivider(),
                const SizedBox(height: Dimensions.space5),
                Text(
                  MyStrings.productsList.tr,
                  style: boldOverLarge.copyWith(
                    color: MyColor.getTextColor(),
                    fontWeight: FontWeight.bold, // Makes title bold
                  ),
                ),
                const SizedBox(height: Dimensions.space10),
                Form(
                  key: formKey,
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      Row(
                        mainAxisAlignment: MainAxisAlignment.spaceBetween,
                        crossAxisAlignment: CrossAxisAlignment.end,
                        children: [
                          Expanded(
                            child: CustomTextField(
                              needOutlineBorder: true,
                              labelText: MyStrings.itemName.tr,
                              controller: controller.itemController,
                              onChanged: (value) {},
                              validator: (value) {
                                if (value!.isEmpty) {
                                  return MyStrings.plsFillOutFieldMsg.tr;
                                } else {
                                  return null;
                                }
                              },
                            ),
                          ),
                          const SizedBox(width: Dimensions.space10),
                          Expanded(
                            child: CustomTextField(
                              textInputType: TextInputType.number,
                              needOutlineBorder: true,
                              labelText: MyStrings.amount.tr,
                              controller: controller.amountController,
                              onChanged: (value) {
                                controller.calculateInvoiceAmount();
                              },
                              validator: (value) {
                                if (value!.isEmpty) {
                                  return MyStrings.plsFillOutFieldMsg.tr;
                                } else {
                                  return null;
                                }
                              },
                            ),
                          ),
                        ],
                      ),
                      SizedBox(height: controller.invoiceItemList.isEmpty ? 0 : Dimensions.space15),
                      ListView.separated(
                        shrinkWrap: true,
                        scrollDirection: Axis.vertical,
                        physics: const NeverScrollableScrollPhysics(),
                        itemCount: controller.invoiceItemList.length,
                        separatorBuilder: (context, index) => const SizedBox(height: Dimensions.space15),
                        itemBuilder: (context, index) {
                          return Row(
                            mainAxisAlignment: MainAxisAlignment.spaceBetween,
                            crossAxisAlignment: CrossAxisAlignment.center,
                            children: [
                              Expanded(
                                child: CustomTextField(
                                  needOutlineBorder: true,
                                  labelText: MyStrings.itemName,
                                  controller: controller.invoiceItemList[index].itemNameController,
                                  onChanged: (value) {},
                                  validator: (value) {
                                    if (value!.isEmpty) {
                                      return MyStrings.plsFillOutFieldMsg.tr;
                                    } else {
                                      return null;
                                    }
                                  },
                                ),
                              ),
                              const SizedBox(width: Dimensions.space10),
                              Expanded(
                                child: CustomTextField(
                                  needOutlineBorder: true,
                                  labelText: MyStrings.amount,
                                  textInputType: TextInputType.number,
                                  controller: controller.invoiceItemList[index].amountController,
                                  onChanged: (value) {
                                    controller.calculateInvoiceAmount();
                                  },
                                  validator: (value) {
                                    if (value!.isEmpty) {
                                      return MyStrings.plsFillOutFieldMsg.tr;
                                    } else {
                                      return null;
                                    }
                                  },
                                ),
                              ),
                              const SizedBox(width: Dimensions.space10),
                              GestureDetector(
                                onTap: () {
                                  controller.decreaseNumberField(index);
                                },
                                child: Container(
                                  padding: const EdgeInsets.all(Dimensions.space5),
                                  margin: const EdgeInsets.only(top: Dimensions.space10),
                                  alignment: Alignment.center,
                                  decoration: BoxDecoration(color: MyColor.colorRed, borderRadius: BorderRadius.circular(Dimensions.space5)),
                                  child: const Icon(Icons.clear, color: MyColor.colorWhite, size: 25),
                                ),
                              )
                            ],
                          );
                        },
                      ),
                      const SizedBox(height: Dimensions.space20),
                      GestureDetector(
                        onTap: () => controller.increaseNumberField(),
                        child: Row(
                          children: [const Icon(Icons.add, color: MyColor.primaryColor, size: 20), const SizedBox(width: Dimensions.space10), Text(MyStrings.addItems.tr, style: regularDefault.copyWith(color: MyColor.primaryColor))],
                        ),
                      ),
                      const SizedBox(height: Dimensions.space20),
                      Column(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Padding(
                            padding: const EdgeInsets.symmetric(vertical: Dimensions.space5),
                            child: RichText(
                              text: TextSpan(
                                children: [
                                  TextSpan(
                                    text: "${MyStrings.totalAmount.tr}: ",
                                    style: regularDefault.copyWith(
                                      fontWeight: FontWeight.bold,
                                      color: MyColor.colorGrey,
                                    ),
                                  ),
                                  TextSpan(
                                    text: "${Get.find<ApiClient>().getCurrencyOrUsername(isSymbol: true)}${controller.totalInvoiceAmount}",
                                    style: regularDefault.copyWith(
                                      color: MyColor.getTextColor(),
                                    ),
                                  ),
                                ],
                              ),
                            ),
                          ),
                          Padding(
                            padding: const EdgeInsets.symmetric(vertical: Dimensions.space5),
                            child: RichText(
                              text: TextSpan(
                                children: [
                                  TextSpan(
                                    text: "${MyStrings.charge.tr}: ",
                                    style: regularDefault.copyWith(
                                      fontWeight: FontWeight.bold,
                                      color: MyColor.colorGrey,
                                    ),
                                  ),
                                  TextSpan(
                                    text: "${Get.find<ApiClient>().getCurrencyOrUsername(isSymbol: true)}${controller.charge}",
                                    style: regularDefault.copyWith(
                                      color: MyColor.getTextColor(),
                                    ),
                                  ),
                                ],
                              ),
                            ),
                          ),
                          Padding(
                            padding: const EdgeInsets.symmetric(vertical: Dimensions.space5),
                            child: RichText(
                              text: TextSpan(
                                children: [
                                  TextSpan(
                                    text: "${MyStrings.getAmount.tr}: ",
                                    style: regularDefault.copyWith(
                                      fontWeight: FontWeight.bold,
                                      color: MyColor.colorGrey,
                                    ),
                                  ),
                                  TextSpan(
                                    text: "${Get.find<ApiClient>().getCurrencyOrUsername(isSymbol: true)}${controller.getableText}",
                                    style: regularDefault.copyWith(
                                      color: MyColor.getTextColor(),
                                    ),
                                  ),
                                ],
                              ),
                            ),
                          ),
                        ],
                      ),
                      const SizedBox(height: Dimensions.space30),
                      RoundedButton(
                          isColorChange: true,
                          text: MyStrings.createInvoice.tr,
                          color: controller.userExist ? MyColor.getPrimaryColor() : MyColor.getTextFieldDisableBorder(),
                          press: controller.userExist == true
                              ? () {
                                  if (formKey.currentState!.validate()) {
                                    FocusScope.of(context).unfocus();
                                    controller.confirmInvoice();
                                  }
                                }
                              : () {})
                    ],
                  ),
                ),
              ],
            ));
  }
}
